%%%%KM-SUB-Skin-Clothing code for making Figure 6a in Lakey at al. Comms. Chem. 2, 56 (2019)

% Reactions included in the model:

% (R1) O3 + Squalene -> (1/3)Acetone + (1/3)6-MHO + (1/3)Geranyl acetone + (1/3)TOT + (1/3)TOP + (1/3)TTT
% (R2) 6-MHO + O3 -> 4-OPA + Acetone
% (R3) Geranyl acetone + O3 -> (1/2)4-OPA + (1/2)4-MON + (1/2)Acetone + (1/2)6-MHO
% (R4) TOP(C22) + O3 ->  (1/4)Acetone + (1/4)6-MHO + (1/4)Geranyl acetone +(1/4)4-OPA + (1/4)4-MOD + (1/4)TTT + (1/4)Product B + (1/4) Product C
% (R5) TOT(C27) + O3 ->  (1/5)Acetone + (1/5)6-MHO + (1/5)Geranyl acetone +(1/5)4-OPA +(1/5)4-MON + (1/5)TTT + (1/5)TOP + (1/5)Product D + (1/5)Product C + (1/5)Product A
% (R6) TTT(C17) + O3 ->  (1/3)Acetone + (1/3)6-MHO + (1/3)Geranyl acetone +(1/3)4-MOD + (1/3)1,4-butanedial + (1/3)Product A
% (R7) 4-MON + O3 -> 2 4-OPA 
% (R8) 4-MOD + O3 -> 4-OPA +1,4 butanedial 
% (R9) O3 + Product A -> (1/2)4-OPA + (1/2)4-MON + (1/2)4-MOD + (1/2)1,4 butanedial
% (R10) O3 + Product B -> 4-MOD + 4-OPA
% (R11) O3 + Product C -> (2/3)4-OPA + (1/3)Product A + (1/3)4-MOD +(1/3)4-MON +(1/3) Product B
% (R12) O3 + Product D -> (1/2)4-OPA +(1/2)4-MON +(1/2)ProductA +(1/2)Product C
% (R13) Other reactive species + O3 -> Products 
% (R14) Laundered species 1 + O3 -> Products
% (R15) Laundered species 2 + O3 -> Products
% Note that the decay of species in the skin are also included using Removal1-5. 
% 
% Species included in the model
%
% i: 1.O3, 
%    4.Squalene, 
%    5.Other reactive species,
%    10. Laundered species 1
%    11. Laundered species 2
%    17.Acetone, 
%    18.6-MHO, 
%    19.Geranyl acetone, 
%    20.TOT, 
%    21.TOP, 
%    22.TTT, 
%    23.4-OPA, 
%    24. 4-MON, 
%    25. 4-MOD, 
%    26.4-butanedial, 
%    27.Product A,
%    28.Product B, 
%    29.Product C, 
%    30.Product D 
%    Note: Additional species can be added into spaces
%    2,3,6,7,8,9,12,13,14,15 and 16 which are currently empty.
%   
%    Layers in the model:
% j: 1. Skin surface (sorption layer)
%    2:oilL+1 Skin oil layers
%    oilL+2:oilL+scL1+1 Top layers of the stratum corneum
%    oilL+scL1+2:oilL+scL1+veL+1 Viable epidermis layers
%    oilL+scL1+veL+2:oilL+scL1+veL+clothesL+1 Gas phase layers in the clothing
%    oilL+scL1+veL+clothesL+2:oilL+scL1+veL+2*clothesL+1 Absorbed phase layer in the clothing
%    oilL+scL1+veL+2*clothesL+2 Blood layer
%    oilL+scL1+veL+2*clothesL+3 Gas phase gap between the clothing and skin
%    oilL+scL1+veL+2*clothesL+4 Mean free path next to the skin surface
%    oilL+scL1+veL+2*clothesL+5:oilL+scL1+veL+2*clothesL+scL2+4 Bottom layers of the stratum corneum
%    (oilL+scL1+veL+2*clothesL+4+scL2)*NCom+2*(i-1)+1 Gas phase
%    (oilL+scL1+veL+2*clothesL+4+scL2)*NCom+2*(i-1)+1 Gas phase next to the clothing
%
%-----------------------------------------------------------------------

clear all;
warning off all;

global scL1 scL2 veL NCom sigma W a0_1 Td_1 kd_1 Ho3 A V_sc1 V_sc2 V_ve Airexchangerate gapR ksc1_sc2
global vekbb scvekbb kaO3 kclothes_gap kclothes_bl_clothes
global ka K Numberofpeople Sizeofroom
global koil_s_o3 koil_s_vol oil_kbb oil_sc_kbb V_oil mfp_vol
global oilL clothesL kclothes_bb Dg V_clothes K_clothing Clothes_fraction_air Clothes_fraction_fibres
global sckbb1 sckbb2 hm  kclothes_gap_mfp S Volume_of_room

%%%%%%%--------------Parameters-----------------%%%%%%%%%%
NCom=30; % Number of species in the model
T=273+25; % Temperature (K)
pi=3.14159;
R_H=82.05746; %Gas constant (cm3 atm K-1 mol-1)
R= 8.31e7; % Gas constant (g cm2 s-2 K-1 mol-1)
NA=6.02e23; %Avogadro's number (mol-1)

%-----------Parameter for O3-----------
Dens=zeros(18); V_w=zeros(18); delta_x=zeros(18);
Dens(1)=1; % Density of ozone (g cm-3)
Mx(1)=48; % Molar mass of ozone (g mol-1)
V_w(1)=Mx(1)/NA/Dens(1);% Volume of one O3 molecule (cm3)
delta_x(1)=V_w(1)^(1/3);% Geometrical diameter of ozone (cm)
sigma(1)=delta_x(1)^2;% Effective molecular cross section of ozone (cm2)
W(1)=sqrt(8*R*T/pi/Mx(1)); %Mean thermal velocity of ozone (cm s-1)

%-----------Parameter for squalene-------------
Dens(4)=0.858; % Density of squalene (g cm-3)
Mx(4)=411; % Molar mass of squalene (g mol-1)
V_w(4)=Mx(4)/NA/Dens(4);% Volume of one squalene molecule (cm3)
delta_x(4)=V_w(4)^(1/3); % Geometrical diameter of squalene (cm)
sigma(4)=delta_x(4)^2;% Effective molecular cross section of squalene (cm2)

%-----------Parameters for Species 17- P30-------------%%%%%%%%

for i=5:18
Dens(i)=1; %Density of species 17 to 30 (g cm-3)
Mx(5)=58; %Molar mass of species 17 (Acetone) (g mol-1)
Mx(6)=126; %Molar mass of species 18 (6-MHO) (g mol-1)
Mx(7)=194; %Molar mass of species 19 (Geranyl acetone) (g mol-1)
Mx(8)=385; %Molar mass of species 20 (TOT) (g mol-1)
Mx(9)= 317; %Molar mass of species 21 (TOP) (g mol-1)
Mx(10)= 248; %Molar mass of species 22 (TTT) (g mol-1)
Mx(11)= 100; %Molar mass of species 23 (4-OPA) (g mol-1)
Mx(12)=168; %Molar mass of species 24 (4-MON) (g mol-1)
Mx(13)= 154; %Molar mass of species 25 (4-MOD) (g mol-1)
Mx(14)=86; %Molar mass of species 26 (1,4 butanedial) (g mol-1)
Mx(15)=222; %Molar mass of species 27 (Product A) (g mol-1)
Mx(16)=222; %Molar mass of species 28 (Product B) (g mol-1)
Mx(17)=290; %Molar mass of species 29 (Product C) (g mol-1)
Mx(18)=359; %Molar mass of species 30 (Product D) (g mol-1)
V_w(i)=Mx(i)/NA/Dens(i); % Volume of one molecule (cm3)
delta_x(i)=V_w(i)^(1/3); %Geometrical diameter (cm)
sigma(i)=delta_x(i)^2; %Effective molecular cross section (cm2)
W(i)=sqrt(8*R*T/pi/Mx(i)); %Mean thermal velocity (cm s-1)
end

%%%%%%----------Geometric factors----------------------%%%%%%%

scL1= 10; %Number of bulk layers in the top of the stratum corneum
scL2=10; %Number of bulk layers in the bottom of the stratum corneum
veL= 10; %Number of bulk layers in the viable epidermis
oilL=5; %Number of bulk layers in the skin oil layer
clothesL=5; %Number of layers in the clothes

scR1= 12.5e-4; %Thickness of the top of the stratum corneum (cm)
scR2= 12.5e-4; %Thickness of the bottom of the stratum corneum (cm)
veR= 100e-4; %Thickness of the viable epidermis (cm)
oilR= 1e-4; %Thickness of the skin oil (cm)
gapR= 0.1; %Thickness of the gap between the clothes and the skin oil (cm)
clothesR=0.1; %Thickness of the clothing (cm)

delta_sc1= (scR1)/scL1; %Thickness of a layer of the top of the stratum corneum (cm)
delta_sc2= (scR2)/scL2; %Thickness of a layer of the bottom of the stratum corneum (cm)
delta_ve= veR/veL; %Thickness of a layer of the viable epidermis (cm)
delta_oil=oilR/oilL; %Thickness of a layer of the skin oil (cm)
delta_clothes=clothesR/clothesL; %Thickness of a layer of the clothing (cm)

A=1; %Surface area being tested 

V_sc1=A*delta_sc1; %Volume of a layer of the top of the stratum corneum (cm3)
V_sc2=A*delta_sc2; %Volume of a layer of the bottom of the stratum corneum (cm3)
V_ve=A*delta_ve; %Volume of a layer of the viable epidermis (cm3)
V_oil=A*delta_oil; %Volume of a layer of the skin oil (cm3)
V_clothes=A*delta_clothes; %Volume of a layer of the clothes (cm3)

%%%%%%------------Kinetic parameters----------------------%%%%%%%

a0_1=0.60; %Surface mass accommodation of all species on an adsorbate free surface
Td_1= 147e-12; %Desorption lifetime of all species on the skin surface (s)
kd_1=1/Td_1; %Desorption rate of all species from the skin surface (s-1)
kaO3=(a0_1*W(1))/4; % Adsorption rate of O3 on an adsorbate free surface (cm s-1)
ka(17)=(a0_1*W(5))/4; % Adsorption rate of acetone on an adsorbate free surface (cm s-1)
ka(18)= (a0_1*W(6))/4; % Adsorption rate of 6-MHO on an adsorbate free surface (cm s-1)
ka(19)= (a0_1*W(7))/4; % Adsorption rate of geranyl acetone on an adsorbate free surface (cm s-1)
ka(23)= (a0_1*W(11))/4; % Adsorption rate of 4-OPA on an adsorbate free surface (cm s-1)
ka(24)= (a0_1*W(12))/4; % Adsorption rate of 4-MON on an adsorbate free surface (cm s-1)
ka(25)= (a0_1*W(13))/4; % Adsorption rate of 4-MOD on an adsorbate free surface (cm s-1)
ka(26)= (a0_1*W(14))/4; % Adsorption rate of 1,4 butanedial on an adsorbate free surface (cm s-1)
Porosity=0.6; %Clothing porosity
Clothes_fraction_air=0.75; % Fraction of the clothing which consists of air
Clothes_fraction_fibres=(1-Clothes_fraction_air)*0.04; %Fraction of the clothing which consists of skin oil and other substances

%%%%%%---------------Diffusion coefficients---------------%%%%%%%
Db_sc=zeros(NCom); Db_ve=zeros(NCom); Db_oil=zeros(NCom); Dg=zeros(NCom);

Db_sc(1)=  1e-8; %Diffusion coefficient of O3 in the stratum corneum (cm2 s-1) 
Db_ve(1)=  1e-7; %Diffusion coefficient of O3 in the viable epidermis (cm2 s-1) 
Db_oil(1)= 3.6e-6; %Diffusion coefficient of O3 in the skin oil (cm2 s-1) 
Dg(1)=0.16; %Gas-phase diffusion coefficient of O3 (cm2 s-1) 

for m=4:NCom
    Db_oil(m)=1e-6;%Diffusion coefficients of species in the skin oil (cm2 s-1)    
end

Dg(17)=0.109; %Gas-phase diffusion coefficient of acetone (cm2 s-1) 
Dg(18)=0.0669; %Gas-phase diffusion coefficient of 6-MHO (cm2 s-1) 
Dg(19)=0.0519; %Gas-phase diffusion coefficient of geranyl acetone (cm2 s-1) 
Dg(23)=0.0818; %Gas-phase diffusion coefficient of 4-OPA (cm2 s-1) 
Dg(24)=0.0584; %Gas-phase diffusion coefficient of 4-MON (cm2 s-1) 
Dg(25)=0.0644; %Gas-phase diffusion coefficient of 4-MOD (cm2 s-1) 
Dg(26)=0.0908; %Gas-phase diffusion coefficient of 1,4-butanedial (cm2 s-1) 

for i=1
    mfp_vol(i)= ((3*Dg(i))/W(i)); %Mean free path of ozone (cm)
end
for i=17:NCom
    mfp_vol(i)= ((3*Dg(i))/W(i-12));%Mean free path of other species (cm)
end

hm(1)=Dg(1)/0.6; %mass- transfer coefficient associated with flux from room air to the clothing for ozone (cm s-1)
for def=17:30
    hm(def)=hm(1)/Dg(1)^(2/3)*Dg(def)^(2/3); %mass- transfer coefficient associated with flux from room air to the clothing (cm s-1)
end

%Stratum corneum diffusion coefficients (cm2 s-1) for:
Db_sc(4)=7.25e-18; %squalene 17
Db_sc(5)=7.25e-18; %other reactive species
Db_sc(17)=4.63e-11; %acetone
Db_sc(18)=9.67e-11; %6-MHO 
Db_sc(19)=1.34e-10; %Geranyl acetone 
Db_sc(20)=1.89e-15; %TOT 
Db_sc(21)=1.49e-13;%TOP 
Db_sc(22)=1.47e-11;%TTT 
Db_sc(23)=5.92e-12;%4-OPA 
Db_sc(24)=2.14e-11;%4-MON 
Db_sc(25)=3.1e-11;%4-MOD 
Db_sc(26)=9.82e-12;%1,4butanedial 
Db_sc(27)=7.15e-11;%Product A 
Db_sc(28)=7.15e-11;%Product B 
Db_sc(29)=1.97e-11;%Product C 
Db_sc(30)=3.36e-13;%Product D 

%Viable epidermis diffusion coefficients (cm2 s-1) for:
Db_ve(4)=3.44e-13; %squalene
Db_ve(5)=3.44e-13; %other reactive species
Db_ve(17)=3.9e-6; %acetone
Db_ve(18)=8.63e-7; %6-MHO 18
Db_ve(19)=9.7e-8; %Geranyl acetone 19
Db_ve(20)=1.32e-11; %TOT 20
Db_ve(21)=2.65e-10;%TOP 21
Db_ve(22)=5.36e-9;%TTT 22
Db_ve(23)=2.97e-6;%4-OPA 23
Db_ve(24)=9.99e-7;%4-MON 24
Db_ve(25)=1.01e-6;%4-MOD 25
Db_ve(26)=3.24e-6;%1,4butanedial 26
Db_ve(27)=1.43e-7;%Product A 27
Db_ve(28)=1.43e-7;%Product B 28
Db_ve(29)=1.01e-8;%Product C 29
Db_ve(30)=5.15e-10;%Product D 30

%%%%%%%--------Partioning coefficients--------%%%%%

%Partitioning coefficients for skin oil (unitless)
Ho3=3.3e-3*R_H*T;%O3
K(17)= 34;% Acetone 
K(18)= 1.26e5; %6-MHO 
K(19)= 3.55e6;% Geranyl acetone 
K(23)= 9.6e3;% 4-OPA 
K(24)= 6.5e6;% 4-MON 
K(25)= 3.25e6;% 4-MOD 
K(26)= 3e3;% 1,4 butanedial 

%Partitioning coefficients for clothing (unitless)
K_clothing(1)=5e-4*R_H*T*1; %O3
K_clothing(17)=34; %Acetone
K_clothing(18)=1.26e5; %6-MHO
K_clothing(19)=3.55e6; %Geranyl acetone
K_clothing(23)=9.6e3; %4-OPA
K_clothing(24)=6.5e6; %4-MON
K_clothing(25)=3.25e6; %4-MOD
K_clothing(26)=3e3; %1,4 butanedial

%%%%%%%---------Kinetic parameters for mass transport-------------%%%%%%%%

koil_s_o3= Db_oil(1)./((delta_x(1)/2)+(delta_oil/2));  % oil to sorption layer transport of o3 (cm s-1)
for i=[17:19,23:26]
koil_s_vol(i)= Db_oil(i)./((delta_x(i-12)/2)+(delta_oil/2)); %oil to sorption layer transport (cm s-1)
end

oil_kbb= Db_oil./delta_oil; %bulk transport in the skin oil (cm s-1)
oil_sc_kbb= ((Db_oil.*delta_oil+Db_sc.*delta_sc1)./(delta_sc1+delta_oil))./((delta_sc1/2)+(delta_oil/2));%transport between the skin oil and the stratum corneum (cm s-1)
ksc1_sc2=Db_sc./(delta_sc1/2+delta_sc2/2); %transport between the top of the lower and the bottom of the higher stratum corneum layers (cm s-1)
    
for i=1:NCom
    kclothes_bb(i)=Dg(i)*Porosity./delta_clothes;%bulk transport in the gas-phase clothing (cm s-1)
    kclothes_gap(i)=Dg(i)./(gapR/2);%bulk transport between the gap and the gas phase in the clothing (cm s-1)
    kclothes_bl_clothes(i)=hm(i)*2;%bulk transport between the boundary layer and the clothing (cm s-1)
    kclothes_gap_mfp(i)=Dg(i)./(gapR/2); %transport between the gap and the mean free path next to the skin (cm s-1)
end

sckbb1= Db_sc./delta_sc1; %bulk transport in the upper stratum corneum (cm s-1)
sckbb2= Db_sc./delta_sc2; %bulk transport in the lower stratum corneum (cm s-1)
vekbb= Db_ve./delta_ve; %bulk transport in the viable epidermis (cm s-1)
scvekbb= ((Db_sc.*delta_sc2+Db_ve.*delta_ve)./(delta_sc2+delta_ve))./((delta_sc2/2)+(delta_ve/2)); %bulk transport between the stratum corneum and viable epidermis (cm s-1)

%%%%%%%%----------------Building parameters------------------------%%%%%%%
%----------------Number of people---------------------------
Numberofpeople=2;
%----------------Size_of_the_room---------------------------
Sizeofroom=30; %m3
%----------------Air exchange rate--------------------------
Airexchangerate=1; %h-1
%-----------Gas phase-------------
S= Numberofpeople*1.7*10000;%total surface area of clothing (cm2)
Volume_of_room=(Sizeofroom-(0.07*Numberofpeople))*1000000 ;%volume of room (cm3)

%%%%%%---------Initial concentrations------------------%%%%%%

y0=zeros(NCom*(oilL+scL1+veL+(2*clothesL)+2+2+scL2)+(2*NCom),1);

%---------Skin oil concentrations------------
for m=1:oilL
y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(4-1)+1+m)=1.5e20;%skin oil concentration of squalene (cm-3)
y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(5-1)+1+m)=1.5e20;%skin oil concentration of other reactive species (cm-3)
end

%---------SC concentrations-----------------
for m=1:scL1
    y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(4-1)+1+oilL+m)=1.5e20/20;%stratum corneum concentration of squalene (cm-3)
    y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(5-1)+1+oilL+m)=1.5e20;%stratum corneum concentration of other reactive species (cm-3)
end
for m=1:scL2
    y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(4-1)+2+oilL+scL1+veL+2+(2*clothesL)+m)=1.5e20/20;%stratum corneum concentration of squalene (cm-3)
    y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(5-1)+2+oilL+scL1+veL+2+(2*clothesL)+m)=1.5e20;%stratum corneum concentration of other reactive species (cm-3)
end

%---------Clothes concentrations condensed phase----------------

for m=1:clothesL
    y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(4-1)+1+oilL+scL1+veL+clothesL+m)=3e18;%squalene concentration (cm-3)
    y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(5-1)+1+oilL+scL1+veL+clothesL+m)=2.2e18; %other reactive species concentration (cm-3)
    y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(10-1)+1+oilL+scL1+veL+clothesL+m)=4e17/2; %laundered species 1 concentration (cm-3)
    y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*(11-1)+1+oilL+scL1+veL+clothesL+m)=3e18/3; %laundered species 2 concentration (cm-3)
end

%----------Gas phase------
y0((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*NCom+1)= 2.46e10*33;% Gas phase concentration of O3 (cm-3)   

%-----------------------------------------------------------------------
%%%%%%%--------Solve differential equations------------%%%%%%%
n=9999; %outputted time steps
t=logspace(-3,log10(3600*4),n);t=t'; %time being investigated (s)
AbsTol=1;
RelTol=1e-3;
options=odeset('AbsTol',AbsTol,'RelTol',RelTol);
[X,Y]=ode23tb(@KMSUB_Skin_Clothing_F,t,y0,options); %solver

%Measurements (these were read from graphs in Wisthaler and Weschler, PNAS, 107, 6568-6575 (2010)

Experimental_data=xlsread('Measurements');

o3_time_1=Experimental_data(:,1); %s
o3_conc_1=Experimental_data(:,2); %ppb
MHO_time_1=Experimental_data(:,3); %s
MHO_conc_1=Experimental_data(:,4); %ppb
GA_conc=MHO_conc_1/8; %ppb
OPA_time_1=Experimental_data(:,5); %s
OPA_conc_1=Experimental_data(:,6); %ppb
Butanedial_conc_1=OPA_conc_1/5; %ppb

%Figures which will be outputted
figure(1)

subplot(5,1,1)    
plot(t/3600,Y(:,((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*NCom+(2*(1-1)+1)))/2.46e10)
hold on
scatter(o3_time_1/3600,o3_conc_1)
title('Ozone')
xlabel('Time (h)')
ylabel('Concentration (ppb)')
hold on
hold off

subplot(5,1,2)
plot(t/3600,Y(:,(((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*NCom+(2*(18-1)+1))))/2.46e10)
hold on
scatter(MHO_time_1/3600,MHO_conc_1)
title('6-MHO')
xlabel('Time (h)')
ylabel('Concentration (ppb)')
hold off

subplot(5,1,3)
plot(t/3600,Y(:,(((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*NCom+(2*(19-1)+1))))/2.46e10)
hold on
scatter(MHO_time_1/3600,GA_conc)
title('Geranyl acetone')
xlabel('Time (h)')
ylabel('Concentration (ppb)')
hold off

subplot(5,1,4)
plot(t/3600,Y(:,(((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*NCom+(2*(23-1)+1))))/2.46e10)
hold on
scatter(OPA_time_1/3600,OPA_conc_1)
title('4-OPA')
xlabel('Time (h)')
ylabel('Concentration (ppb)')
hold off

subplot(5,1,5)
plot(t/3600,Y(:,(((oilL+scL1+veL+(2*clothesL)+2+2+scL2)*NCom+(2*(26-1)+1))))/2.46e10)
hold on
scatter(OPA_time_1/3600,Butanedial_conc_1)
title('1,4 butanedial')
xlabel('Time (h)')
ylabel('Concentration (ppb)')
hold off
